/*******************************************************************************
*                                                                              *
* Project       : VAnet                                                        *
* Filename      : VAnetPodule.cpp                                              *
* Version       : 0.01 (25-Jul-2002)                                           *
* Author        : David J Ruck                                                 *
* Description   : TCP Socket support on Virtual Acorn / Red Squirrel Emulator  *
* Information   : Windows DLL source file                                      *
*                                                                              *
********************************************************************************
* Change Log:                                                                  *
*                                                                              *
* Ver   Date         Description of change                                     *
* ----  -----------  ---------------------                                     *
* 0.01  25-Jul-2002  Initial revision                                          *
*                                                                              *
*******************************************************************************/

//#include "stdafx.h"

// unfortunate clash...
#undef LoadString

#include <stdint.h>
#include "podules-win.h"
//#include "rpcemu.h"
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "sys/stat.h"
#include "Winsock2.h"
//#include "Console.h"
#include "VAnetPodule.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

FILE *vanetlogf;

void vanetlog(const char *format, ...)
{
   char buf[1024];
//return;
   if (!vanetlogf) vanetlogf=fopen("valog.txt","wt");
   va_list ap;
   va_start(ap, format);
   vsprintf(buf, format, ap);
   va_end(ap);
   fputs(buf,vanetlogf);
   fflush(vanetlogf);
}
extern "C" {
extern __declspec(dllexport) int InitDll();
extern __declspec(dllexport) uint8_t  readb(podule *p, int easi, uint32_t addr);
extern __declspec(dllexport) uint16_t readw(podule *p, int easi, uint32_t addr);
extern __declspec(dllexport) uint32_t readl(podule *p, int easi, uint32_t addr);
extern __declspec(dllexport) void writeb(podule *p, int easi, uint32_t addr, uint8_t val);
extern __declspec(dllexport) void writew(podule *p, int easi, uint32_t addr, uint16_t val);
extern __declspec(dllexport) void writel(podule *p, int easi, uint32_t addr, uint32_t val);
extern __declspec(dllexport) int timercallback(podule *p);
}

//////////////////////////////////////////////////////////////////////
// types
//////////////////////////////////////////////////////////////////////

// RISC OS structure is subtly different
struct hostent2
{
	char*   h_name;
	char**  h_aliases;
	int     h_addrtype;
	int     h_length;
	char**  h_addr_list;
};

//////////////////////////////////////////////////////////////////////
// Utility functions
//////////////////////////////////////////////////////////////////////

const char *ipaddr(struct sockaddr_in *paddr)
{
	static char address[STRING_LEN];

	sprintf(address, "%s:%d", inet_ntoa((struct in_addr)paddr->sin_addr),
		                      ntohs(paddr->sin_port));

	return address;
}

//////////////////////////////////////////////////////////////////////
// DLL Entry
//////////////////////////////////////////////////////////////////////

// Standard windoisms, only needed if mfc is used (e.g. property panels)

//static AFX_EXTENSION_MODULE NEAR extensionDLL = { NULL, NULL };
/*
BOOL APIENTRY DllMain( HINSTANCE hModule,
                       DWORD  ul_reason_for_call,
                       LPVOID lpReserved)
{
	if (ul_reason_for_call == DLL_PROCESS_ATTACH)
	{
		// Extension DLL one-time initialization - do not allocate memory here,
		//   use the TRACE or ASSERT macros or call MessageBox
		if (!AfxInitExtensionModule(extensionDLL, hModule))
			return 0;

		// Other initialization could be done here, as long as
		// it doesn't result in direct or indirect calls to AfxGetApp.
		// This extension DLL doesn't need to access the app object
		// but to be consistent with testdll1.dll, this DLL requires
		// explicit initialization as well (see below).

		// This allows for greater flexibility later in development.
	}
	return TRUE;
}*/

// Exported DLL initialization is run in context of running application
/*extern "C" void __declspec(dllexport) __cdecl InitMFCDll()
{
	// create a new CDynLinkLibrary for this app
	new CDynLinkLibrary(extensionDLL);
	// nothing more to do
}*/

// callback handler for timer event.
//static Word timerevent_dispatch(IEvent*e, int delay)
//{
//	return ((CVAnetPodule*)e->iface.impl)->TimerEvent(delay);
//}

//////////////////////////////////////////////////////////////////////
// CVAnetPodule class
//////////////////////////////////////////////////////////////////////
// Construction/Destruction

// declare the component so it can be created from a model file
//IMPLEMENT_COMPONENT("VAnetPodule", CVAnetPodule);

// Constructor. Only basic initialisation should be done here
CVAnetPodule::CVAnetPodule()
{
//        rpclog("Constructor!\n");
	mRomBase     = NULL;
	mRomSize     = 0;
	mIntMask     = 0;
	mLastCommand = 0;
	mpRomPage    = (Word*)(mRegisters+PODULE_ROMPAGE);
	mpIntStatus  = (Word*)(mRegisters+PODULE_INTR);
	mpCommand    = (Word*)(mRegisters+PODULE_COMMAND);
	mpReturn     = (Word*)(mRegisters+PODULE_RETURN);
	mParams      = (Word*)(mRegisters+PODULE_PARAMS);
	mpBlock	     = (Byte*)(mRegisters+PODULE_BLOCK);

	mLastEventCode   = -1;
	mLastEventSocket = -1;
	mLastEventIP     = 0;
		
	memset(mRegisters, 0, PODULE_REGSIZE);

	for(int s=0; s<MAX_SOCKETS; s++)
	{
		mSockets[s].s = INVALID_SOCKET;
		mSockets[s].f = 0;
	}

	FD_ZERO(&mAsyncSet);

//	mTimer.iface.size   = sizeof(IEvent);
//	mTimer.iface.type   = "VANetPoduleTimer";	// not exported, just give it a friendly name
//	mTimer.iface.impl   = this;
//	mTimer.ProcessEvent = timerevent_dispatch;
//	mScheduler          = NULL;
}

CVAnetPodule::~CVAnetPodule()
{
}

// In init we perform local tasks such as allocating memory
BOOL CVAnetPodule::Init()
{
	// Load the rom
	const char* romname = "VAnetRom.dat";//LoadString("rom", "VAnetRom.dat");
	char        path[1024];

	FILE* fp = 0;
//        rpclog("Loading ROM\n");
//	for (const char** spath = CComponentManager::GetSearchPath(); *spath; ++spath)
//	{
//		strcpy(path, *spath);
		strcpy(path, romname);

		fp = fopen(path, "rb");
		if (!fp)
                   return FALSE;
	if (fp)
	{
		fseek(fp, 0, SEEK_END);
		mRomSize = ftell(fp);
		fseek(fp, 0, SEEK_SET);

		mRomBase = new Byte[mRomSize];
		fread(mRomBase, 1, mRomSize, fp);
		fclose(fp);
	}
	else
	{
//		rpclog("VAnetPodule failed to load rom image\n");
		return FALSE;
	}

	WORD    wVersionRequested = MAKEWORD( 2, 2 );;
	int     err               = WSAStartup(wVersionRequested, &mWsaData);
	if ( err != 0 )
	{
//		rpclog("VAnetPodule failed to initialise WinSock\n");
		return FALSE;
	}
//	rpclog("We're done!\n");
        return TRUE;
//	return CIocPodule::Init();
}

// In connect we connect to other components
BOOL CVAnetPodule::Connect()
{
        return TRUE;
/*	mScheduler = (IScheduler*)FindInterface(IFACE_SCHEDULER);
	if (!mScheduler)
		return FALSE;

	mTimerTicks = mScheduler->TickRate(mScheduler) / TIMER_RATE;
	mScheduler->ScheduleEvent(mScheduler, &mTimer, mTimerTicks);
	
	return CIocPodule::Connect();*/
}

BOOL CVAnetPodule::Disconnect()
{
        return TRUE;
/*	if (mScheduler)
		mScheduler->DescheduleEvent(mScheduler, &mTimer);

	mScheduler = NULL;

	return CIocPodule::Disconnect();*/
}

BOOL CVAnetPodule::Cleanup()
{
	WSACleanup();

	for(int i=0; i<MAX_SOCKETS; i++)
	{
		closesocket(mSockets[i].s);
		mSockets[i].s = INVALID_SOCKET;
	}

	delete[] mRomBase;
	mRomBase  = NULL;
	mRomSize  = 0;
        return TRUE;
//	return CIocPodule::Cleanup();
}

int firstvanetwrite=0;
// Handle byte writes to our podule.
void CVAnetPodule::WriteByte(podule *p, Word address, Byte b)
{
        if (!firstvanetwrite)
        {
                firstvanetwrite=1;
                p->msectimer=10;
                vanetlog("Firstvanet msectimer\n");
        }
        vanetlog("Write byte %08X %02X\n",address,b);
	if(address>=PODULE_REGISTERS && address<PODULE_REGISTERS+PODULE_REGSIZE)
	{
		Word reg = address-PODULE_REGISTERS;
		mRegisters[reg] = b;
		Trigger(p, reg, true);
	}
}

// Handle Word writes to our podule.
void CVAnetPodule::WriteWord(podule *p, Word address, Word w)
{
        vanetlog("Write word %08X %02X\n",address,w);
	if(address>=PODULE_REGISTERS && address<PODULE_REGISTERS+PODULE_REGSIZE)
	{
		Word reg = address-PODULE_REGISTERS;
		*(Word*)(mRegisters+reg) = w;
		Trigger(p, reg, true);
	}
}

// Handle byte reads from our podule
Byte CVAnetPodule::ReadByte(podule *p, Word address)
{
//        vanetlog("Read byte %08X\n",address);
//        rpclog("Read %08X %08X ",address,mRomBase);
	if(address>=PODULE_REGISTERS && address<PODULE_REGISTERS+PODULE_REGSIZE)
	{
		Word reg = address-PODULE_REGISTERS;
//                rpclog("Reg\n");
		Trigger(p, reg, false);
		return mRegisters[address-PODULE_REGISTERS];
	}
	else if (mRomBase && address<PODULE_REGISTERS)
	{
//                rpclog("Dat\n");
		Word romaddr = ((address>>2)&ROM_PAGE_MASK) + ((*mpRomPage)<<ROM_PAGE_SHIFT);
		return mRomBase[romaddr];
	}
	else
	{
//                rpclog("Nothing\n");
		return 0xff;
	}
}

// handle Word reads from our podule
Word CVAnetPodule::ReadWord(podule *p, Word address)
{
//        vanetlog("Read word %08X\n",address);
	if(address>=PODULE_REGISTERS && address<PODULE_REGISTERS+PODULE_REGSIZE)
	{
		Word reg = address-PODULE_REGISTERS;
		Trigger(p, reg, false);
		return *(Word*)(mRegisters+address-PODULE_REGISTERS);
	}
	else
	{
		// ROM is only read with byte operations
		return ReadByte(p,address);
	}
}


// trigger action on writable locations
void CVAnetPodule::Trigger(podule *p, Word reg, bool write)
{
        vanetlog("Trigger %i %i\n",reg,write);
	if(write)
	{
		switch(reg)
		{
			case PODULE_ROMPAGE:
				if (((*mpRomPage)<<ROM_PAGE_SHIFT) >= mRomSize)
					*mpRomPage = 0;
				break;

			case PODULE_INTR:
				mIntMask     = *mpIntStatus;
				*mpIntStatus = 0;
				p->irq=0;//ClearIrq();
				break;

			case PODULE_COMMAND:
				StartCommand();
				break;

			default:
				break;
		}
	}
	else
	{
		if(reg==PODULE_RETURN && *mpReturn==RET_WAIT)
			ExecCommand();
	}
}

// execute command
void CVAnetPodule::StartCommand(void)
{
        vanetlog("Start command %08X\n",*mpCommand);
	*mpReturn = RET_WAIT;

	switch(*mpCommand)
	{
		case 0:	// initialisation call
			{
				// copy hostname to block +4
				mParams[0] = 0;
				mParams[1] = 4;
				char *name = (char*)(mpBlock+mParams[1]);
				if(gethostname(name, 256)!=0)
				{
					Error(1u, "gethostname failed [%d]", WSAGetLastError());
					return;
				}
				else
				{
					mParams[2] = (mParams[1]+strlen(name)+3) & ~3;

					// got host address to block+0
					struct hostent *h = gethostbyname(name);
					if(h)
					{
						memcpy(mpBlock+mParams[0], h->h_addr_list[0], sizeof(long));
					}
					else
					{
						Error(2u, "gethostbyname did not return an entry");
						return;
					}
				}
				// command completed
				*mpReturn = RET_DONE;
			}
			break;

		case Socket_Creat:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Creat(%d, %d, %d)", mParams[0], mParams[1], mParams[2]);
#endif
			break;

		case Socket_Bind:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Bind(%d, %s, %d)",
				            mParams[0],
				            ipaddr((struct sockaddr_in*)(mpBlock+mParams[1])),
							mParams[2]);
#endif
			CheckSockAddr((struct sockaddr_in*)(mpBlock+mParams[1]));
			break;

		case Socket_Listen:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Listen(%d, %d)", mParams[0], mParams[1]);
#endif
			break;

		case Socket_Accept:
		case Socket_Accept_1:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Accept%s(%d, ",
				            *mpCommand==Socket_Accept_1 ? "_1":"",
							mParams[0]);
#endif
			break;

		case Socket_Connect:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Connect(%d, %s, %d)",
				            mParams[0],
							ipaddr((struct sockaddr_in*)(mpBlock+mParams[1])),
							mParams[2]);
#endif
			// Convert BSD 4.4 to 4.3
			CheckSockAddr((struct sockaddr_in*)(mpBlock+mParams[1]));
			break;

		case Socket_Read:
			mParams[3] = 0;
			// no break;
		case Socket_Recv:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_%s(%d, block+&%X, %d, %d)",
				            *mpCommand==Socket_Read ? "Read":"Recv",
				            mParams[0], mParams[1], mParams[2], mParams[3]);
#endif
			break;

		case Socket_Recvfrom:
		case Socket_Recvfrom_1:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Recvfrom%s(%d, block+&%X, %d, %d,",
				            *mpCommand==Socket_Recvfrom_1 ? "_1":"",
				             mParams[0], mParams[1], mParams[2], mParams[3]);
#endif
			break;

		case Socket_Write:
			mParams[3] = 0;
			// no break;
		case Socket_Send:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_%s(%d, block+&%X, %d, %d)",
				            *mpCommand==Socket_Read ? "Write":"Send",
				            mParams[0], mParams[1], mParams[2], mParams[3]);
#endif
			break;

		case Socket_Sendto:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Sendto(%d, block+&%X, %d, %d, %s, %d)",
				            mParams[0], mParams[1], mParams[2], mParams[3],
							ipaddr((struct sockaddr_in*)(mpBlock+mParams[4])),
							mParams[5]);
#endif
			CheckSockAddr((struct sockaddr_in*)(mpBlock+mParams[4]));
			break;

		case Socket_Shutdown:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Shutdown(%d, %d)", mParams[0], mParams[1]);
#endif
			break;

		case Socket_Setsockopt:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Setsockopt(%d, %d, %d, block+&%X, %d)",
				            mParams[0], mParams[1], mParams[2], mParams[3], mParams[4]);
#endif
			// downgrade unsupported BSD 4.4 call
			if(mParams[1]==SOL_SOCKET && mParams[2]==SO_REUSEPORT)
				mParams[2] = SO_REUSEADDR;
			break;

		case Socket_Getsockopt:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Getsockopt(%d, %d, %d, block+&%X, %d",
				            mParams[0], mParams[1], mParams[2], mParams[3], mParams[4]);
#endif
			break;

		case Socket_Getpeername:
		case Socket_Getpeername_1:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Getpeername%s(%d, %s, %d",
				            *mpCommand==Socket_Getpeername_1 ? "_1":"",
				            mParams[0],
							ipaddr((struct sockaddr_in*)(mpBlock+mParams[1])),
							mParams[2]);
#endif
			CheckSockAddr((struct sockaddr_in*)(mpBlock+mParams[1]));
			break;

		case Socket_Getsockname:
		case Socket_Getsockname_1:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Getsockname%s(%d, %s, %d",
				            *mpCommand==Socket_Getsockname_1 ? "_1":"",
				            mParams[0],
							ipaddr((struct sockaddr_in*)(mpBlock+mParams[1])),
							mParams[2]);
#endif
			CheckSockAddr((struct sockaddr_in*)(mpBlock+mParams[1]));
			break;

		case Socket_Close:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Close[%d]", mParams[0]);
#endif
			break;

		case Socket_Select:
			{
				// Set timeout for polled operation
/*				mTimeout = clock();

				if(mParams[4]!=-1)
				{
					struct timeval* tv =(struct timeval*)(mpBlock+mParams[4]);
					mTimeout          += (tv->tv_sec  * CLOCKS_PER_SEC)
					                  +  (tv->tv_usec * CLOCKS_PER_SEC)/1000000;
				}*/
			}
			break;

		case Socket_Ioctl:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Ioctl(%d, &%08X, block+&%X)",
			                mParams[0], mParams[1], mParams[2]);
#endif
			switch(mParams[1])
			{
				case FIOSLEEPTW:
					// ignore taskwindow sleep and complete
					mParams[0] = 0;
					*mpReturn  = RET_DONE;
					break;

				case FIOASYNC:
					// handle async internally
					if(*(DWORD*)(mpBlock+mParams[2])!=0)
					{
						FD_SET(mSockets[mParams[0]].s, &mAsyncSet);
						mSockets[mParams[0]].f |= SOCKFLAG_ASYNC;
					}
					else
					{
						FD_CLR(mSockets[mParams[0]].s, &mAsyncSet);
						mSockets[mParams[0]].f &= ~SOCKFLAG_ASYNC;
					}
					// complete call
					mParams[0] = 0;
					*mpReturn  = RET_DONE;
					break;
				case FIONBIO:
					// flag socket as non blocking / blocking
					if(*(DWORD*)(mpBlock+mParams[2])!=0)
						mSockets[mParams[0]].f &= ~SOCKFLAG_BLOCKING;
					else
						mSockets[mParams[0]].f |= SOCKFLAG_BLOCKING;
					break;
					
				default:
					// handle in ExecCommand
					break;
			}
#ifdef SOCKET_TRACE
			if(*mpReturn == RET_DONE)
				console->printf("\n");
#endif
			break;

		case Socket_Stat:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Stat(%d, block+&%X)", mParams[0], mParams[1]);
#endif
			break;

		case Socket_Gettsize:
			mParams[0] = MAX_SOCKETS < mWsaData.iMaxSockets ? MAX_SOCKETS : mWsaData.iMaxSockets;
#ifdef SOCKET_TRACE
			console->printf("VAnet::Socket_Gettsize() = %d\n", mParams[0]);
#endif
			*mpReturn = RET_DONE;
			break;

		case Socket_Sendtosm:
		case Socket_InternalLookup:
		case Socket_Version:
			// internal to internet module
			Error(*mpCommand, "Not yet implemented by VAnet Plugin");
			break;

		case Socket_Recvmsg:
		case Socket_Recvmsg_1:
		case Socket_Sendmsg:
		case Socket_Sendmsg_1:
			Error(*mpCommand, "Should be handled by VAInternet module");
			break;

		case Socket_Readv:
		case Socket_Writev:
		case Socket_Sysctl:
			Error(*mpCommand, "Not yet implemented by VAnet Plugin");
			break;

		case Resolver_GetHostByName:
#ifdef SOCKET_TRACE
			console->printf("VAnet::Resolver_GetHostByName(\"%s\")", (char*)(mpBlock+mParams[1]));
#endif
			break;

		case Resolver_GetHost:
			if(mParams[0]==-1)
			{
				// convert old GetHostByAddr call from address to address string
				mParams[0] = 0;
				strcpy((char*)mpBlock+mParams[0], inet_ntoa(*(struct in_addr*)(mpBlock+mParams[1])));
			}
#ifdef SOCKET_TRACE
			console->printf("VAnet::Resolver_GetHost(\"%s\")", (char*)(mpBlock+mParams[0]));
#endif
			break;

		case Resolver_GetCache:
		case Resolver_CacheControl:
			Error(*mpCommand, "Not yet implemented by VAnet Plugin");
			break;

		default:
//			ASSERT(true);
			break;
	}

	if(*mpReturn == RET_WAIT)
		ExecCommand();
}

void CVAnetPodule::ExecCommand(void)
{
	int wsaerr = 0;
	int result = 0;
	
	vanetlog("ExecCommand %08X\n",*mpCommand);

	WSASetLastError(0);

	switch(*mpCommand)
	{
		case Socket_Creat:
			result = AddSocket(socket(mParams[0], mParams[1], mParams[2]));
			break;

		case Socket_Bind:
			result = bind(mSockets[mParams[0]].s, (struct sockaddr*)(mpBlock+mParams[1]), mParams[2]);
			break;

		case Socket_Listen:
			result = listen(mSockets[mParams[0]].s, mParams[1]);
			break;

		case Socket_Accept:
		case Socket_Accept_1:
			result = AddSocket(accept(mSockets[mParams[0]].s, (struct sockaddr*)(mpBlock+mParams[1]), (int*)&mParams[2]));
			if(result!=INVALID_SOCKET)
			{
				mSockets[result].f = (mSockets[result].f     & ~SOCKFLAG_BLOCKING)
				                   | (mSockets[mParams[0]].f &  SOCKFLAG_BLOCKING);
			}
			break;

		case Socket_Connect:
			result = connect(mSockets[mParams[0]].s, (struct sockaddr*)(mpBlock+mParams[1]), mParams[2]);
			break;

		case Socket_Read:
		case Socket_Recv:
			result = recv(mSockets[mParams[0]].s, (char*)(mpBlock+mParams[1]), mParams[2], mParams[3]);
			break;

		case Socket_Recvfrom:
		case Socket_Recvfrom_1:
			result = recvfrom(mSockets[mParams[0]].s, (char*)(mpBlock+mParams[1]), mParams[2], mParams[3], (struct sockaddr*)(mpBlock+mParams[4]), (int*)&mParams[5]);
			break;

		case Socket_Write:
		case Socket_Send:
			result = send(mSockets[mParams[0]].s, (char*)(mpBlock+mParams[1]), mParams[2], mParams[3]);
			break;

		case Socket_Sendto:
			result = sendto(mSockets[mParams[0]].s, (char*)(mpBlock+mParams[1]), mParams[2], mParams[3], (struct sockaddr*)(mpBlock+mParams[4]), mParams[5]);
			break;

		case Socket_Shutdown:
			result = shutdown(mSockets[mParams[0]].s, mParams[1]);
			break;

		case Socket_Setsockopt:
			result = setsockopt(mSockets[mParams[0]].s, mParams[1], mParams[2], (char*)(mpBlock+mParams[3]), mParams[4]);
			break;

		case Socket_Getsockopt:
			result = getsockopt(mSockets[mParams[0]].s, mParams[1], mParams[2], (char*)(mpBlock+mParams[3]), (int*)&mParams[4]);
			break;

		case Socket_Getpeername:
		case Socket_Getpeername_1:
			result = getpeername(mSockets[mParams[0]].s, (struct sockaddr*)(mpBlock+mParams[1]), (int*)&mParams[2]);
			break;

		case Socket_Getsockname:
		case Socket_Getsockname_1:
			result = getsockname(mSockets[mParams[0]].s, (struct sockaddr*)(mpBlock+mParams[1]), (int*)&mParams[2]);
			break;

		case Socket_Close:
			result = closesocket(mSockets[mParams[0]].s);
			break;

		case Socket_Select:
			{
				int ndesc = mParams[0] < MAX_SOCKETS ? mParams[0] : MAX_SOCKETS;
				struct timeval zto;
				fd_set fdsets[3];

				zto.tv_sec  = 0;
				zto.tv_usec = 0;

				MakeFDSets(ndesc, fdsets);

				// issue select with no timeout
				result = select(ndesc,
								mParams[1]==-1 ? NULL : &fdsets[0],
								mParams[2]==-1 ? NULL : &fdsets[1],
								mParams[3]==-1 ? NULL : &fdsets[2],
								&zto);

				// if no activity and before timeout
				// exit so that we will be called again
				if(result==0)// && clock()<mTimeout)
					return;

				// decode the result
				if(result>=0)
					DecodeFDSets(ndesc, fdsets);
			}
			break;

		case Socket_Ioctl:
			result = ioctlsocket(mSockets[mParams[0]].s, mParams[1], (DWORD*)(mpBlock+mParams[2]));
			break;

		case Socket_Stat:
			result = _fstat(mSockets[mParams[0]].s, (struct _stat*)mParams[1]);
			break;

		case Resolver_GetHostByName:
			result    = (int)gethostbyname((char*)(mpBlock+mParams[1]));
			break;

		case Resolver_GetHost:
			result    = (int)gethostbyname((char*)(mpBlock+mParams[0]));
			break;

		default:
//			ASSERT(true);
			break;
	}

	wsaerr = WSAGetLastError();

	FinishCommand(result, wsaerr);
}

void CVAnetPodule::FinishCommand(int result, int wsaerr)
{
	bool err_on_r0 = true;
	
	vanetlog("FinishCommand %i %i %08X\n",result,wsaerr,*mpCommand);

	switch(*mpCommand)
	{
		case Socket_Creat:
		case Socket_Bind:
		case Socket_Listen:
		case Socket_Connect:
		case Socket_Read:
		case Socket_Recv:
		case Socket_Write:
		case Socket_Send:
		case Socket_Sendto:
		case Socket_Shutdown:
		case Socket_Setsockopt:
		case Socket_Ioctl:
		case Socket_Stat:
#ifdef SOCKET_TRACE
			console->printf(" = %d [%d]\n", result, wsaerr);
#endif
			break;

		case Socket_Accept:
		case Socket_Accept_1:
		case Socket_Getpeername:
		case Socket_Getpeername_1:
		case Socket_Getsockname:
		case Socket_Getsockname_1:
#ifdef SOCKET_TRACE
			console->printf("=%s, =%d) = %d [%d]\n",
				            ipaddr((struct sockaddr_in*)(mpBlock+mParams[1])),
							mParams[2], result, wsaerr);
#endif
			break;

		case Socket_Recvfrom:
		case Socket_Recvfrom_1:
#ifdef SOCKET_TRACE
			console->printf(" =%s, =%d) = %d [%d]\n",
				            ipaddr((struct sockaddr_in*)(mpBlock+mParams[4])),
							mParams[5], result, wsaerr);
#endif
			break;

		case Socket_Getsockopt:
#ifdef SOCKET_TRACE
			console->printf("=%d) = %d [%d]\n", mParams[4], result, wsaerr);
#endif
			break;

		case Socket_Close:
				// close the socket and release the array entry for reuse
				if(mSockets[mParams[0]].f & SOCKFLAG_ASYNC)
					FD_CLR(mSockets[mParams[0]].s, &mAsyncSet);

				mSockets[mParams[0]].s = INVALID_SOCKET;
				mSockets[mParams[0]].f = 0;
#ifdef SOCKET_TRACE
			console->printf(" = %d [%d]\n", mParams[0], wsaerr);
#endif
			break;

		case Socket_Select:
#ifdef SOCKET_TRACE
			if(mLastCommand!=Socket_Select || result!=0)
				TraceSelect(result, wsaerr);
#endif
			break;

		case Resolver_GetHostByName:
		case Resolver_GetHost:
			err_on_r0 = false;
			result    = ReturnHostent(result, wsaerr);
			break;

		default:
//			ASSERT(true);
			break;
	}

	mParams[0] = result;

	if(err_on_r0 && mParams[0]==-1)
		Error((wsaerr>10000 && wsaerr<10100) ? wsaerr-10000 : wsaerr, "Socket error %d", wsaerr);
	else
		*mpReturn = RET_DONE;

	mLastCommand = *mpCommand;
}

// return a host end structure in the memory block
int CVAnetPodule::ReturnHostent(int result, int wsaerr)
{
	struct hostent  *hent  = (struct hostent*)result;
	struct hostent2 *hent2 = (struct hostent2*)mpBlock;

	if(hent==NULL)
	{ 
		result     = (wsaerr==WSAHOST_NOT_FOUND) ? -1 : (wsaerr>10000 && wsaerr<10100) ? wsaerr-10000 : wsaerr; 
		mParams[1] = -1;
#ifdef SOCKET_TRACE
	console->printf(" = %d [%d]\n", result, wsaerr);
#endif
		return result;
	}

	result      = 0;
	mParams[1]  = 0;
	int offset  = sizeof(struct hostent2);
	int aliases = 0;
	int addrs   = 0;
	int i       = 0;
	int len     = 0;

	// copy in to block
	hent2->h_addrtype  = hent->h_addrtype;
	hent2->h_length    = hent->h_length; 

	// make room for the address pointers
	addrs              = offset;
	hent2->h_addr_list = (char**)offset;
	i                  = 0;
	do
	{
		offset += 4;
	}
	while(hent->h_addr_list[i++] && offset<PODULE_BLOCK_SIZE-8);

	// make room for the aliases pointers
	aliases          = offset;
	hent2->h_aliases = (char**)offset;
	i                = 0;
	do
	{
		offset += 4;
	}
	while(hent->h_aliases[i++]!=0 && offset<PODULE_BLOCK_SIZE);

	// copy addreses
	i = 0;
	do
	{
		if(hent->h_addr_list[i])
		{
			*(DWORD*)(mpBlock+addrs) = offset;
			memcpy((void*)(mpBlock+offset), hent->h_addr_list[i], hent->h_length);
			addrs  += 4;
			offset += hent->h_length;
		}
		else
		{
			*(DWORD*)(mpBlock+addrs) = 0;
		}
	}
	while(hent->h_addr_list[i++] && offset<PODULE_BLOCK_SIZE-8);

	// copy name
	hent2->h_name = (char*)offset;
	len           = (strlen(hent->h_name)+1+3) & ~3;
	if(offset+len<PODULE_BLOCK_SIZE)
		strcpy((char*)(mpBlock+offset), hent->h_name);
	offset       += len;

	// copy alias strings, updating the pointers
	i = 0;
	do
	{
		if(hent->h_aliases[i])
		{
			len = strlen(hent->h_aliases[i])+1;

			if(offset+len<PODULE_BLOCK_SIZE)
			{
				*(DWORD*)(mpBlock+aliases) = offset;
				strcpy((char*)(mpBlock+offset), hent->h_aliases[i]);
			}
			else
			{
				*(DWORD*)(mpBlock+aliases) = 0;
			}
			aliases += 4;
			offset  += len;
		}
		else
		{
			*(DWORD*)(mpBlock+aliases) = 0;
		}
	} while(hent->h_aliases[i++]!=0);

#ifdef SOCKET_TRACE
	console->printf(" = %s [%d]\n",
		            mParams[1]!=-1 ? inet_ntoa(*(struct in_addr*)hent->h_addr) : "NULL",
					wsaerr);
#endif

	return result;
}

// A timer event happend
int CVAnetPodule::TimerEvent(podule *p)
{
	fd_set rset = mAsyncSet;
	fd_set eset = mAsyncSet;
	struct timeval zto;
	
//	vanetlog("Timerevent!\n");
	zto.tv_sec  = 0;
	zto.tv_usec = 0;

	if(select(MAX_SOCKETS, &rset, NULL, &eset, &zto)>0)
	{
		DWORD *pEventCode   = (DWORD*)(mpBlock+PODULE_EVENT_CODE);
		DWORD *pEventSocket = (DWORD*)(mpBlock+PODULE_EVENT_SOCKET);
		DWORD *pEventIP     = (DWORD*)(mpBlock+PODULE_EVENT_IP);

		for(int s=0; s<MAX_SOCKETS; s++)
		{
			if(mSockets[s].s!=INVALID_SOCKET)
			{
				if(FD_ISSET(mSockets[s].s, &eset))
				{
					*pEventCode   = 2;
					*pEventSocket = s;
					*pEventIP     = 0;
				}
				else if(FD_ISSET(mSockets[s].s, &rset))
				{
					*pEventCode   = 1;
					*pEventSocket = s;
					*pEventIP     = 0;
				}
				// #### not not yet handling event codes 3 or 4
			}
		}

		// prevent repeated IRQs for the same event
		if(  *pEventCode!=mLastEventCode   ||
		   *pEventSocket!=mLastEventSocket ||
		       *pEventIP!=mLastEventIP)
		{
			// generate interrupt if mask set to non zero
			if((*mpIntStatus = mIntMask) != 0)
                          p->irq=1;
//				SetIrq();

			mLastEventCode   = *pEventCode;
			mLastEventSocket = *pEventSocket;
			mLastEventIP     = *pEventIP;
		}
	}
	else
	{
		// clear last event code on no activity
		mLastEventCode = -1;
	}

	// carry on at the same rate
	return 10;//mTimerTicks;
}

void __cdecl CVAnetPodule::Error(Word errnum, const char *message, ...)
{
	*mpReturn = RET_ERROR;
	*(Word*)(mpBlock+0) = errnum;
	va_list	vp;
	va_start(vp, message);
	vsprintf((char*)(mpBlock+4), message, vp);
	va_end(vp);
};

#ifdef SOCKET_TRACE
void CVAnetPodule::TraceSelect(int result, int wsaerr)
{
	console->printf("VAnet::Socket_Select(%d, ", mParams[0]);

	for(int i=1; i<=3; i++)
	{
		if(mParams[i]!=-1)
			console->printf("block+&%X=&%08X%08X, ", mParams[i], *(DWORD*)(mpBlock+mParams[i]+4), *(DWORD*)(mpBlock+mParams[i]+0));
		else
			console->printf("NULL, ");
	}

	if(mParams[4]==-1)
		console->printf("NULL)");
	else
		console->printf("%d.%05d)", ((struct timeval*)(mpBlock+mParams[4]))->tv_sec, ((struct timeval*)(mpBlock+mParams[4]))->tv_usec);

	console->printf(" = %d [%d]\n", result, wsaerr);
}
#endif

// Must map windows socket handles to array index of 0 - 255 for RISC OS
SOCKET CVAnetPodule::AddSocket(SOCKET s)
{
        int i;
	if(s == INVALID_SOCKET)
		return s;

	// Find first free index in array, dont allocate 0
	for(i=1; i<MAX_SOCKETS && mSockets[i].s!=INVALID_SOCKET; i++)
	{ /* empty */ }

	if(i==MAX_SOCKETS)
	{
		// no free indexes
		closesocket(s);
		WSASetLastError(WSAEMFILE);
		return INVALID_SOCKET;
	}
	else
	{
		// assign to array and mark as blocking until client calls ioctl
		mSockets[i].s = s;
		mSockets[i].f = SOCKFLAG_BLOCKING;
		return (SOCKET)i;
	}
}

// translate RISC OS one bit per socket fd_set's to Winsock format
void CVAnetPodule::MakeFDSets(int ndesc, FD_SET *fdsets)
{
	for(int j=0; j<3; j++)
	{
		FD_ZERO(&fdsets[j]);

		if(mParams[j+1] != -1)
		{
			for(int i=0; i<ndesc; i++)
			{
				if(( ((Byte*)(mpBlock+mParams[j+1]))[i>>3] & (1<<(i&7)) )!=0 && mSockets[i].s!=INVALID_SOCKET)
					FD_SET(mSockets[i].s, &fdsets[j]);
			}
		}
	}
}

// convert WinSock fd_set's back to RISC OS format
void CVAnetPodule::DecodeFDSets(int ndesc, FD_SET *fdsets)
{
	for(int j=0; j<3; j++)
	{
		if(mParams[j+1] != -1)
		{
			memset(mpBlock+mParams[j+1], 0, (ndesc+7)>>3);

			for(int i=0; i<ndesc; i++)
			{
				if(mSockets[i].s!=INVALID_SOCKET && FD_ISSET(mSockets[i].s, &fdsets[j]))
					((Byte*)(mpBlock+mParams[j+1]))[i>>3] |= (1<<(i&7));
			}
		}
	}
}

// checks a sockaddr_in is not in BSD4.4 format - Winsock only accepts BSD4.3
void CVAnetPodule::CheckSockAddr(struct sockaddr_in* saddr)
{
	if((saddr->sin_family & 0xFF)==0)
	{
		saddr->sin_family = (saddr->sin_family >> 8) & 0xFF;
	}
}

CVAnetPodule cvanet;

int InitDll()
{
        cvanet.Init();
//        rpclog("Network inited!\n");
}

uint8_t readb(podule *p, int easi, uint32_t addr)
{
        return cvanet.ReadByte(p,addr);
//        rpclog("Read Podule B %08X\n",addr);
//        if (!(addr&0xC000))
//        return 0xFF;
}
uint16_t readw(podule *p, int easi, uint32_t addr)
{
        return cvanet.ReadWord(p,addr);
}

uint32_t readl(podule *p, int easi, uint32_t addr)
{
        return cvanet.ReadWord(p,addr);
//        rpclog("Read Podule L %08X\n",addr);
//        if (!(addr&0xC000))
//        return 0xFFFFFFFF;
}
void writeb(podule *p, int easi, uint32_t addr, uint8_t val)
{
//        rpclog("Write Podule B %08X %08X\n",addr,val);
        cvanet.WriteByte(p,addr,val);
}
void writew(podule *p, int easi, uint32_t addr, uint16_t val)
{
        cvanet.WriteWord(p,addr,val);
}
void writel(podule *p, int easi, uint32_t addr, uint32_t val)
{
//        rpclog("Write Podule L %08X %08X\n",addr,val);
        cvanet.WriteWord(p,addr,val);
}

int timercallback(podule *p)
{
//        vanetlog("timercallback\n");
        return cvanet.TimerEvent(p);
}
